%%%%%%%%%%%%%%%%%%gausselimfp%%%%%%%%%%%%%%%
%This function performs Gaussian elimination on the input matrix A and the 
%vector b to solve for y in the system Ay=b.  To improve numerical
%stability, it utilizes full pivoting.
%Note that this code is NOT intelligent about numerical error (e.g. it 
%doesn't know that 10^(-12) should probably be 0).  If the input matrix 
%is analytically singular but has a numerical solution (due to numerical 
%error), this code will give the numerical solution, not the analytical one.
%
%Inputs:
%A: an nxn matrix
%b: a nx1 vector
%
%Outputs:
%y: the nx1 solution to the system Ay=b

function y=gausselimfp(A,b)
%We first need the sizes of our input matricies
[n,n1]=size(A);
[n2,n3]=size(b);

%We next do some error checking to make sure the inputs make sense
if n~=n1
    error('A is not square')
elseif  n~=n2
    error('The number of rows in b does not equal the number of rows in A')
elseif n3~=1
    error('b has more than one column')
end

%Swapping columns permutes the solution vector x.  We will store these
%permutations here.  This vector stores permutations like (2 3),
%which would, in this case, switch columns 2 and 3.  Since it was
%initialized to ones, if no columns are switched in a step of
%pivoting, the permutation for the step will merely be the identity
%permutation (1 1)
permvector=ones(n,2);

%%%%%%%%%%%%%%%%%%%%%%Forward Elimination%%%%%%%%%%%%%%%%%%%%%
%Here we perform the pivoting and forward elimination to reduce the matrix
%to an upper triangular matrix (assuming it is non-singular)
for it=1:n-1
    %%%%%%%%%%%%%%%%%Performing Full Pivoiting%%%%%%%%%%%%%%%%%%%
    %First we find the largest value in the lower right hand corner
     bigvalue=[0,0,0];
    for i=it:n
        for j=it:n
            if abs(A(i,j))>abs(bigvalue(1,1))
                bigvalue=[A(i,j) i j];
            end
        end
    end
    %If the search doesn't find anything, the matrix is singular
    if (bigvalue(1,2)==0) && (bigvalue(1,3)==0)
        error('A is singular')
    end
    
    %First we swap/pivot the rows
    if bigvalue(1,2)~=it
        temp1=A(it,:);
        A(it,:)=A(bigvalue(1,2),:);
        A(bigvalue(1,2),:)=temp1;
        %Recall we need to swap the rows of b too
        temp2=b(it,1);
        b(it,1)=b(bigvalue(1,2),1);
        b(bigvalue(1,2),1)=temp2;
    end
    
    %Next we swap/pivot the columns
    if bigvalue(1,3)~=it
        temp1=A(:,it);
        A(:,it)=A(:,bigvalue(1,3));
        A(:,bigvalue(1,3))=temp1;
        %We don't need to swap anything in b, but we do need to record the
        %permutation of the vector x
        permvector(it,:)=[it bigvalue(1,3)];
    end
    
    %One Step of Gaussian Elimination
    for i=it+1:n
        b(i,1)=b(i,1)-b(it,1)/A(it,it)*A(i,it);
        A(i,:)=A(i,:)-A(it,:)/A(it,it)*A(i,it);
    end

end


%Since we only performed Gaussian elimination on the 1:n-1'th rows, we
%need to check if the n'th row is zero.  If so, the matrix is singular.
if A(n,n)==0
    error('A is singular')
end

%%%%%%%%%%%%%%%%%%%%Backwards Substitution%%%%%%%%%%%%
%Next we perform backwards substitution to obtain our solution.  Note that
%it is unnecessary for obtaining the solution to overwrite any of the 
%entries in A in this step
for j=n:-1:2
    for i=j-1:-1:1
        b(i,1)=b(i,1)-b(j,1)/A(j,j)*A(i,j);
    end
    b(j,1)=b(j,1)/A(j,j);
end
b(1,1)=b(1,1)/A(1,1);




%%%%%%%%%%%%%%%%Applying Permutations%%%%%%%%%%%%%%
%Recall that whenever we swap the columns of A, we permute the entries of
%our x solution vector.  Here we undo those permutations, recalling that
%the permutations must be applied in backwards order
for i=n:-1:1
    temp1=b(permvector(i,1));
    b(permvector(i,1))=b(permvector(i,2));
    b(permvector(i,2))=temp1;
end

%The vector b is our solution y
y=b;

